
import { useState } from 'react';
import { Navigation } from '@/components/Navigation';
import { Footer } from '@/components/Footer';
import { motion } from 'framer-motion';
import { Calendar } from "@/components/ui/calendar";
import { Button } from '@/components/ui/button';
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { toast } from "sonner";

const Reservation = () => {
  const [lang, setLang] = useState<'en' | 'ar'>('en');
  const [date, setDate] = useState<Date | undefined>(new Date());

  const content = {
    en: {
      title: "Make a Reservation",
      subtitle: "Book Your Table",
      form: {
        name: "Full Name",
        email: "Email Address",
        phone: "Phone Number",
        guests: "Number of Guests",
        time: "Preferred Time",
        submit: "Reserve Table"
      },
      success: "Reservation submitted successfully!"
    },
    ar: {
      title: "احجز طاولة",
      subtitle: "احجز طاولتك",
      form: {
        name: "الاسم الكامل",
        email: "البريد الإلكتروني",
        phone: "رقم الهاتف",
        guests: "عدد الضيوف",
        time: "الوقت المفضل",
        submit: "احجز الآن"
      },
      success: "تم تقديم الحجز بنجاح!"
    }
  };

  const timeSlots = [
    "12:00", "12:30", "13:00", "13:30", "14:00", "14:30",
    "18:00", "18:30", "19:00", "19:30", "20:00", "20:30", "21:00"
  ];

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    toast.success(content[lang].success);
  };

  return (
    <div className="min-h-screen" dir={lang === 'ar' ? 'rtl' : 'ltr'}>
      <Navigation lang={lang} onLanguageChange={setLang} />
      <div className="pt-16">
        <div className="container section-padding">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="max-w-4xl mx-auto"
          >
            <h1 className="text-4xl md:text-5xl font-serif font-bold text-primary text-center mb-16">
              {content[lang].title}
            </h1>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
              <div className="bg-accent rounded-lg p-6">
                <Calendar
                  mode="single"
                  selected={date}
                  onSelect={setDate}
                  className="rounded-md border"
                  disabled={(date) => date < new Date()}
                />
              </div>

              <form onSubmit={handleSubmit} className="space-y-6">
                <Input 
                  type="text" 
                  placeholder={content[lang].form.name}
                  required 
                />
                <Input 
                  type="email" 
                  placeholder={content[lang].form.email}
                  required 
                />
                <Input 
                  type="tel" 
                  placeholder={content[lang].form.phone}
                  required 
                />
                <Select>
                  <SelectTrigger>
                    <SelectValue placeholder={content[lang].form.guests} />
                  </SelectTrigger>
                  <SelectContent>
                    {[1,2,3,4,5,6,7,8].map(num => (
                      <SelectItem key={num} value={num.toString()}>
                        {num} {lang === 'en' ? 'Guests' : 'ضيوف'}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Select>
                  <SelectTrigger>
                    <SelectValue placeholder={content[lang].form.time} />
                  </SelectTrigger>
                  <SelectContent>
                    {timeSlots.map(time => (
                      <SelectItem key={time} value={time}>
                        {time}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Button type="submit" className="w-full">
                  {content[lang].form.submit}
                </Button>
              </form>
            </div>
          </motion.div>
        </div>
      </div>
      <Footer lang={lang} />
    </div>
  );
};

export default Reservation;
