import { Navigation } from '@/components/Navigation';
import { Footer } from '@/components/Footer';
import { useState, useRef, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import {
  Dialog,
  DialogContent,
  DialogTrigger,
} from "@/components/ui/dialog";

const Menu = () => {
  // Get the language from URL parameters
  const [lang, setLang] = useState<'en' | 'ar'>(() => {
    const params = new URLSearchParams(window.location.search);
    return params.get('lang') === 'ar' ? 'ar' : 'en';
  });
  const [activeCategory, setActiveCategory] = useState<string>('');
  const categoryRefs = useRef<{ [key: string]: HTMLDivElement | null }>({});

  const handleLanguageChange = (newLang: 'en' | 'ar') => {
    setLang(newLang);
    const params = new URLSearchParams(window.location.search);
    params.set('lang', newLang);
    window.history.replaceState({}, '', `${window.location.pathname}?${params.toString()}`);
  };

  const content = {
    en: {
      title: "Our Menu",
      categories: [
        {
          name: "Main Courses",
          items: [
            { 
              name: "Grilled Salmon", 
              price: "$24", 
              description: "Fresh salmon with herbs and lemon",
              image: "https://images.unsplash.com/photo-1467003909585-2f8a72700288?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "Beef Tenderloin", 
              price: "$28", 
              description: "Premium cut with red wine sauce",
              image: "https://images.unsplash.com/photo-1546964124-0cce460f38ef?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "Chicken Supreme", 
              price: "$22", 
              description: "Herb-crusted chicken breast",
              image: "https://images.unsplash.com/photo-1604908176997-125f25cc6f3d?auto=format&fit=crop&w=800&q=80"
            }
          ]
        },
        {
          name: "Appetizers",
          items: [
            { 
              name: "Caesar Salad", 
              price: "$12", 
              description: "Classic salad with our special dressing",
              image: "https://images.unsplash.com/photo-1546793665-c74683f339c1?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "Soup of the Day", 
              price: "$8", 
              description: "Ask your server for today's selection",
              image: "https://images.unsplash.com/photo-1547592166-23ac45744acd?auto=format&fit=crop&w=800&q=80"
            }
          ]
        },
        {
          name: "Desserts",
          items: [
            { 
              name: "Chocolate Fondant", 
              price: "$10", 
              description: "Warm chocolate cake with vanilla ice cream",
              image: "https://images.unsplash.com/photo-1624353365286-3f8d62daad51?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "Crème Brûlée", 
              price: "$9", 
              description: "Classic French dessert",
              image: "https://images.unsplash.com/photo-1470324161839-ce2bb6fa6bc3?auto=format&fit=crop&w=800&q=80"
            }
          ]
        }
      ]
    },
    ar: {
      title: "قائمتنا",
      categories: [
        {
          name: "الأطباق الرئيسية",
          items: [
            { 
              name: "سلمون مشوي", 
              price: "$24", 
              description: "سلمون طازج مع الأعشاب والليمون",
              image: "https://images.unsplash.com/photo-1467003909585-2f8a72700288?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "لحم بقري تندرلوين", 
              price: "$28", 
              description: "قطعة ممتازة مع صلصة النبيذ الأحمر",
              image: "https://images.unsplash.com/photo-1546964124-0cce460f38ef?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "دجاج سوبريم", 
              price: "$22", 
              description: "صدر دجاج مغطى بالأعشاب",
              image: "https://images.unsplash.com/photo-1604908176997-125f25cc6f3d?auto=format&fit=crop&w=800&q=80"
            }
          ]
        },
        {
          name: "المقبلات",
          items: [
            { 
              name: "سلطة سيزر", 
              price: "$12", 
              description: "سلطة كلاسيكية مع صلصتنا الخاصة",
              image: "https://images.unsplash.com/photo-1546793665-c74683f339c1?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "حساء اليوم", 
              price: "$8", 
              description: "اسأل النادل عن اختيار اليوم",
              image: "https://images.unsplash.com/photo-1547592166-23ac45744acd?auto=format&fit=crop&w=800&q=80"
            }
          ]
        },
        {
          name: "الحلويات",
          items: [
            { 
              name: "فوندان الشوكولاتة", 
              price: "$10", 
              description: "كعكة شوكولاتة دافئة مع آيس كريم الفانيليا",
              image: "https://images.unsplash.com/photo-1624353365286-3f8d62daad51?auto=format&fit=crop&w=800&q=80"
            },
            { 
              name: "كريم بروليه", 
              price: "$9", 
              description: "حلوى فرنسية كلاسيكية",
              image: "https://images.unsplash.com/photo-1470324161839-ce2bb6fa6bc3?auto=format&fit=crop&w=800&q=80"
            }
          ]
        }
      ]
    }
  };

  const scrollToCategory = (categoryName: string) => {
    const element = categoryRefs.current[categoryName];
    if (element) {
      element.scrollIntoView({ behavior: 'smooth' });
      setActiveCategory(categoryName);
    }
  };

  useEffect(() => {
    const handleScroll = () => {
      const categories = content[lang].categories;
      for (const category of categories) {
        const element = categoryRefs.current[category.name];
        if (element) {
          const rect = element.getBoundingClientRect();
          if (rect.top <= 200 && rect.bottom >= 200) {
            setActiveCategory(category.name);
            break;
          }
        }
      }
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, [lang]);

  return (
    <div className="min-h-screen" dir={lang === 'ar' ? 'rtl' : 'ltr'}>
      <Navigation lang={lang} onLanguageChange={handleLanguageChange} />
      <div className="pt-16">
        <div className="sticky top-16 bg-white/80 backdrop-blur-sm z-40 shadow-sm">
          <div className="container mx-auto px-4 py-4">
            <div className="flex gap-4" style={{ direction: lang === 'ar' ? 'rtl' : 'ltr' }}>
              {content[lang].categories.map((category) => (
                <button
                  key={category.name}
                  onClick={() => scrollToCategory(category.name)}
                  className={`px-4 py-2 rounded-full transition-all whitespace-nowrap font-arabic
                    ${activeCategory === category.name 
                      ? 'bg-primary text-white shadow-md transform scale-105' 
                      : 'bg-gray-100 hover:bg-gray-200'}`}
                >
                  {category.name}
                </button>
              ))}
            </div>
          </div>
        </div>

        <div className="container mx-auto px-4 py-12">
          <motion.h1 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="text-4xl font-serif font-bold text-primary mb-12 text-center"
          >
            {content[lang].title}
          </motion.h1>
          
          <div className="space-y-24" style={{ perspective: '2000px' }}>
            {content[lang].categories.map((category, index) => (
              <motion.div
                key={index}
                ref={el => categoryRefs.current[category.name] = el}
                initial={{ opacity: 0, y: 50 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true, margin: "-100px" }}
                transition={{ duration: 0.5 }}
                className="bg-white rounded-lg shadow-md p-6"
                style={{
                  transform: `rotateX(5deg) translateZ(${index * -100}px)`,
                  transformStyle: "preserve-3d",
                }}
              >
                <h2 className="text-2xl font-serif font-bold text-primary mb-6">
                  {category.name}
                </h2>
                <div className="grid gap-6">
                  {category.items.map((item, itemIndex) => (
                    <motion.div
                      key={itemIndex}
                      initial={{ opacity: 0, x: lang === 'ar' ? 20 : -20 }}
                      whileInView={{ opacity: 1, x: 0 }}
                      viewport={{ once: true }}
                      transition={{ delay: itemIndex * 0.1 }}
                      className="border-b border-gray-200 pb-4 last:border-0"
                    >
                      <div className="flex gap-4">
                        <Dialog>
                          <DialogTrigger>
                            <div className="w-24 h-24 rounded-lg overflow-hidden flex-shrink-0 hover:opacity-90 transition-opacity">
                              <img 
                                src={item.image} 
                                alt={item.name}
                                className="w-full h-full object-cover"
                              />
                            </div>
                          </DialogTrigger>
                          <DialogContent className="max-w-2xl">
                            <div className="aspect-video w-full overflow-hidden rounded-lg">
                              <img 
                                src={item.image} 
                                alt={item.name}
                                className="w-full h-full object-cover"
                              />
                            </div>
                            <div className="mt-4">
                              <h3 className="text-xl font-semibold">{item.name}</h3>
                              <p className="text-gray-600 mt-2">{item.description}</p>
                              <p className="text-primary font-semibold mt-2">{item.price}</p>
                            </div>
                          </DialogContent>
                        </Dialog>
                        <div className="flex-grow">
                          <div className="flex justify-between items-center mb-2">
                            <h3 className="text-lg font-semibold">{item.name}</h3>
                            <span className="text-primary font-semibold">{item.price}</span>
                          </div>
                          <p className="text-gray-600">{item.description}</p>
                        </div>
                      </div>
                    </motion.div>
                  ))}
                </div>
              </motion.div>
            ))}
          </div>
        </div>
      </div>
      <Footer lang={lang} />
    </div>
  );
};

export default Menu;
