
import { useState } from 'react';
import { Navigation } from '@/components/Navigation';
import { Footer } from '@/components/Footer';
import { motion } from 'framer-motion';
import {
  Dialog,
  DialogContent,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";

const Events = () => {
  const [lang, setLang] = useState<'en' | 'ar'>('en');

  const content = {
    en: {
      title: "Our Events & Gallery",
      subtitle: "Capturing Memorable Moments",
      categories: [
        {
          id: "dinners",
          label: "Special Dinners",
          images: [
            {
              title: "Anniversary Dinner",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            },
            {
              title: "Holiday Feast",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            }
          ]
        },
        {
          id: "cooking",
          label: "Cooking Classes",
          images: [
            {
              title: "Traditional Cooking Workshop",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            },
            {
              title: "Kids Cooking Class",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            }
          ]
        },
        {
          id: "cultural",
          label: "Cultural Events",
          images: [
            {
              title: "Heritage Festival",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            },
            {
              title: "Music Night",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            }
          ]
        }
      ]
    },
    ar: {
      title: "الفعاليات والمعرض",
      subtitle: "لحظات لا تنسى",
      categories: [
        {
          id: "dinners",
          label: "العشاء الخاص",
          images: [
            {
              title: "عشاء الذكرى السنوية",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            },
            {
              title: "وليمة العطلات",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            }
          ]
        },
        {
          id: "cooking",
          label: "دروس الطبخ",
          images: [
            {
              title: "ورشة الطبخ التقليدي",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            },
            {
              title: "فصل طبخ للأطفال",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            }
          ]
        },
        {
          id: "cultural",
          label: "الفعاليات الثقافية",
          images: [
            {
              title: "مهرجان التراث",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            },
            {
              title: "ليلة موسيقية",
              image: "https://images.unsplash.com/photo-1514933651103-005eec06c04b?q=80&w=1974&auto=format&fit=crop"
            }
          ]
        }
      ]
    }
  };

  return (
    <div className="min-h-screen" dir={lang === 'ar' ? 'rtl' : 'ltr'}>
      <Navigation lang={lang} onLanguageChange={setLang} />
      <div className="pt-16">
        <div className="container section-padding">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="text-center mb-16"
          >
            <h1 className="text-4xl md:text-5xl font-serif font-bold text-primary mb-4">
              {content[lang].title}
            </h1>
            <p className="text-xl text-secondary">
              {content[lang].subtitle}
            </p>
          </motion.div>

          <Tabs defaultValue="dinners" className="w-full">
            <TabsList className="flex justify-center mb-8">
              {content[lang].categories.map((category) => (
                <TabsTrigger key={category.id} value={category.id}>
                  {category.label}
                </TabsTrigger>
              ))}
            </TabsList>

            {content[lang].categories.map((category) => (
              <TabsContent key={category.id} value={category.id}>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                  {category.images.map((item, index) => (
                    <motion.div
                      key={index}
                      initial={{ opacity: 0, y: 20 }}
                      whileInView={{ opacity: 1, y: 0 }}
                      viewport={{ once: true }}
                      transition={{ delay: index * 0.1 }}
                    >
                      <Dialog>
                        <DialogTrigger className="w-full">
                          <div className="group relative overflow-hidden rounded-lg aspect-[4/3]">
                            <img 
                              src={item.image} 
                              alt={item.title}
                              className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-110"
                            />
                            <div className="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex items-center justify-center">
                              <h3 className="text-white text-xl font-semibold">
                                {item.title}
                              </h3>
                            </div>
                          </div>
                        </DialogTrigger>
                        <DialogContent className="max-w-4xl">
                          <div className="aspect-video w-full overflow-hidden rounded-lg">
                            <img 
                              src={item.image} 
                              alt={item.title}
                              className="w-full h-full object-cover"
                            />
                          </div>
                          <h3 className="text-xl font-semibold mt-4">
                            {item.title}
                          </h3>
                        </DialogContent>
                      </Dialog>
                    </motion.div>
                  ))}
                </div>
              </TabsContent>
            ))}
          </Tabs>
        </div>
      </div>
      <Footer lang={lang} />
    </div>
  );
};

export default Events;
