
import { useState } from 'react';
import { Navigation } from '@/components/Navigation';
import { Footer } from '@/components/Footer';
import { motion } from 'framer-motion';
import { MapPin, Phone, Mail, Clock } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { toast } from "sonner";

const Contact = () => {
  const [lang, setLang] = useState<'en' | 'ar'>('en');

  const content = {
    en: {
      title: "Contact Us",
      subtitle: "Get in Touch",
      info: {
        address: "123 Restaurant Street, City Name",
        phone: "+1 234 567 890",
        email: "info@beitmeryam.com",
        hours: "Daily: 11:00 AM - 10:00 PM"
      },
      form: {
        name: "Full Name",
        email: "Email Address",
        message: "Your Message",
        submit: "Send Message"
      }
    },
    ar: {
      title: "اتصل بنا",
      subtitle: "تواصل معنا",
      info: {
        address: "١٢٣ شارع المطعم، اسم المدينة",
        phone: "+١ ٢٣٤ ٥٦٧ ٨٩٠",
        email: "info@beitmeryam.com",
        hours: "يومياً: ١١:٠٠ ص - ١٠:٠٠ م"
      },
      form: {
        name: "الاسم الكامل",
        email: "البريد الإلكتروني",
        message: "رسالتك",
        submit: "إرسال الرسالة"
      }
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    toast.success(lang === 'en' ? "Message sent successfully!" : "تم إرسال الرسالة بنجاح!");
  };

  return (
    <div className="min-h-screen" dir={lang === 'ar' ? 'rtl' : 'ltr'}>
      <Navigation lang={lang} onLanguageChange={setLang} />
      <div className="pt-16">
        <div className="container section-padding">
          <motion.h1 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="text-4xl md:text-5xl font-serif font-bold text-primary text-center mb-16"
          >
            {content[lang].title}
          </motion.h1>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
            <motion.div 
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              className="space-y-8"
            >
              <h2 className="text-2xl font-serif font-bold text-secondary">
                {content[lang].subtitle}
              </h2>
              
              <div className="space-y-6">
                <div className="flex items-center gap-4">
                  <MapPin className="w-6 h-6 text-primary flex-shrink-0" />
                  <p className="text-secondary">{content[lang].info.address}</p>
                </div>
                <div className="flex items-center gap-4">
                  <Phone className="w-6 h-6 text-primary flex-shrink-0" />
                  <p className="text-secondary">{content[lang].info.phone}</p>
                </div>
                <div className="flex items-center gap-4">
                  <Mail className="w-6 h-6 text-primary flex-shrink-0" />
                  <p className="text-secondary">{content[lang].info.email}</p>
                </div>
                <div className="flex items-center gap-4">
                  <Clock className="w-6 h-6 text-primary flex-shrink-0" />
                  <p className="text-secondary">{content[lang].info.hours}</p>
                </div>
              </div>
            </motion.div>

            <motion.form 
              initial={{ opacity: 0, x: 20 }}
              animate={{ opacity: 1, x: 0 }}
              onSubmit={handleSubmit}
              className="space-y-6"
            >
              <Input 
                type="text" 
                placeholder={content[lang].form.name}
                required 
              />
              <Input 
                type="email" 
                placeholder={content[lang].form.email}
                required 
              />
              <Textarea 
                placeholder={content[lang].form.message}
                className="min-h-[150px]"
                required 
              />
              <Button type="submit" className="w-full">
                {content[lang].form.submit}
              </Button>
            </motion.form>
          </div>
        </div>
      </div>
      <Footer lang={lang} />
    </div>
  );
};

export default Contact;
