import { useState } from 'react';
import { Menu, X, Languages } from 'lucide-react';
import { Button } from '@/components/ui/button';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";

interface NavigationProps {
  lang: 'en' | 'ar';
  onLanguageChange?: (lang: 'en' | 'ar') => void;
}

export const Navigation = ({ lang, onLanguageChange }: NavigationProps) => {
  const [isOpen, setIsOpen] = useState(false);

  const getLanguageUrl = (baseUrl: string) => {
    return lang === 'ar' ? `${baseUrl}?lang=ar` : baseUrl;
  };

  const content = {
    en: {
      leftLinks: [
        { text: "Home", href: "/" },
        { text: "Menu", href: "/menu" },
        { text: "About", href: "/about" },
      ],
      rightLinks: [
        { text: "Events", href: "/events" },
        { text: "Contact", href: "/contact" },
        { text: "Reservation", href: "/reservation" },
      ],
    },
    ar: {
      leftLinks: [
        { text: "الرئيسية", href: "/" },
        { text: "القائمة", href: "/menu" },
        { text: "من نحن", href: "/about" },
      ],
      rightLinks: [
        { text: "الفعاليات", href: "/events" },
        { text: "اتصل بنا", href: "/contact" },
        { text: "احجز", href: "/reservation" },
      ],
    }
  };

  return (
    <nav className="fixed top-0 left-0 w-full z-50 bg-white/90 backdrop-blur-sm shadow-sm" dir={lang === 'ar' ? 'rtl' : 'ltr'}>
      <div className="container mx-auto px-4">
        <div className="flex items-center justify-between h-16">
          <Button
            variant="ghost"
            size="icon"
            className="md:hidden z-20"
            onClick={() => setIsOpen(!isOpen)}
            aria-expanded={isOpen}
          >
            {isOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
          </Button>

          {/* Left Navigation - Hidden on Mobile */}
          <div className={`hidden md:flex items-center gap-8`}>
            {content[lang].leftLinks.map((link) => (
              <a
                key={link.href}
                href={getLanguageUrl(link.href)}
                className="text-foreground/80 hover:text-primary transition-colors px-2 py-1"
              >
                {link.text}
              </a>
            ))}
          </div>

          {/* Logo - Center */}
          <div className="absolute left-1/2 transform -translate-x-1/2 flex items-center justify-center">
            <a href={getLanguageUrl('/')} className="flex flex-col items-center">
              <img 
                src="/lovable-uploads/d36179cb-9d9c-401f-af62-27d44714efcb.png" 
                alt={lang === 'ar' ? 'مطعم بيت مريم' : 'Beit Meryam Restaurant'}
                className="h-12 w-auto"
              />
            </a>
          </div>

          {/* Right Navigation - Hidden on Mobile */}
          <div className={`hidden md:flex items-center gap-8`}>
            {content[lang].rightLinks.map((link) => (
              <a
                key={link.href}
                href={getLanguageUrl(link.href)}
                className="text-foreground/80 hover:text-primary transition-colors px-2 py-1"
              >
                {link.text}
              </a>
            ))}
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost" size="icon">
                  <Languages className="h-5 w-5" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align={lang === 'ar' ? 'start' : 'end'}>
                <DropdownMenuItem onClick={() => onLanguageChange?.('en')}>
                  English
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => onLanguageChange?.('ar')}>
                  العربية
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        </div>
      </div>

      {/* Mobile Navigation */}
      {isOpen && (
        <div className="md:hidden bg-white border-t absolute w-full left-0 top-16 z-10">
          <div className="container mx-auto px-4 py-4">
            {[...content[lang].leftLinks, ...content[lang].rightLinks].map((link) => (
              <a
                key={link.href}
                href={getLanguageUrl(link.href)}
                className="block py-3 text-foreground/80 hover:text-primary transition-colors text-lg"
                onClick={() => setIsOpen(false)}
              >
                {link.text}
              </a>
            ))}
          </div>
        </div>
      )}
    </nav>
  );
};
